/*
 * Copyright (c) Doug Palmer <doug@charvolant.org> 2005
 *
 * See LICENSE for licensing details.
 * 
 * $Id$
 */

package org.charvolant.sudoku;

import java.util.*;

/**
 * A single cell in the sudoku grid.
 * <p>
 * This contains a value binding, if there is one, and
 * an array of possible values.
 * 
 * @author doug
 *
 */
public class Cell extends Model {
  /** The special value that indicates a non-fixed value: {@value} */
  private static final int NOT_FIXED = -1;
  
  /** The parent board */
  private Board board;
  /** The shapes that this cell is part of */
  private Set<Shape> shapes;
  /** The array of possible values */
  private boolean[] possibilities;
  /** What is the fixed value */
  private int value;
  /** Does this have a single possible value? */
  private int singleton;
  
  /**
   * Construct for a parent board
   * 
   * @param board The parent board
   */
  public Cell(Board board) {
    this.board = board;
    this.shapes = new HashSet<Shape>(3);
    this.reset();
  }

  /**
   * Add a shape to the list of shapes that
   * constrain this cell.
   * 
   * @param shape The shape to add
   */
  public void addShape(Shape shape) {
    this.shapes.add(shape);
    shape.addCell(this);
  }
  
  /**
   * Get the square size for a cell.
   * <p>
   * This can be used to lay out possibilities in
   * the cell.
   *
   * @return The square size.
   *
   * @see org.charvolant.sudoku.Board#getSquareSize()
   */
  public int getSquareSize() {
    return this.board.getSquareSize();
  }

  /**
   * Get the X position of a possibility.
   * <p>
   * This is laid out in the same way as a square.
   *
   * @param sn The square number
   * 
   * @return The X position
   *
   * @see org.charvolant.sudoku.Board#getSquareX(int)
   */
  public int getSquareX(int sn) {
    return this.board.getSquareX(sn);
  }

  /**
   * Get the Y position of a possibility.
   * <p>
   * This is laid out in the same way as a square.
   *
   * @param sn The square number
   * 
   * @return The Y position
   *
   * @see org.charvolant.sudoku.Board#getSquareY(int)
   */
  public int getSquareY(int sn) {
    return this.board.getSquareY(sn);
  }

  /**
   * Get the game context.
   * 
   * @see org.charvolant.sudoku.Board#getContext()
   * 
   * @return The context
   */
  public Context getContext() {
    return this.board.getContext();
  }

  /**
   * Is this cell fixed?
   *
   * @return Returns true if fixed.
   */
  public boolean isFixed() {
    return this.value != this.NOT_FIXED;
  }
  
  /**
   * Is this a singleton cell?
   * <p>
   * This is true if the cell is not fixed but has only one possibility.
   * 
   * @return Returns true if this cell is a singleton
   */
  public boolean isSingleton() {
    int pcount = 0;
    
    if (this.isFixed())
      return false;
    if (this.singleton != this.NOT_FIXED)
      return true;
    for (int i = 0; i < this.possibilities.length; i++)
      if (this.possibilities[i])
        pcount++;
    return pcount == 1;
  }
  
  /**
   * Is this an empty cell?
   * <p>
   * This is true if there are no possibilites left.
   * 
   * @return True if there are no possibilities.
   */
  public boolean isEmpty() {
    int pcount = 0;
    
    if (this.isFixed())
      return false;
    for (int i = 0; i < this.possibilities.length; i++)
      if (this.possibilities[i])
        pcount++;
    return pcount == 0; 
  }

  /**
   * Is this a possible value?
   * 
   * @param value The value
   * 
   * @return True if this value is still possible in this cell
   */
  public boolean isPossible(int value) {
    return this.possibilities[value];
  }
  
  /**
   * Get the possibilities.
   * <p>
   * This is an array that has a true at a spot if the
   * value is a possibility.
   *
   * @return Returns the possibilities.
   */
  public boolean[] getPossibilities() {
    return this.possibilities;
  }

  /**
   * Get the fixed value.
   *
   * @return Returns the fixed value.
   */
  public int getValue() {
    return this.value;
  }

  /**
   * Get the singleton value.
   *
   * @return Returns the singleton value.
   */
  public int getSingleton() {
    int count = 0, single = this.NOT_FIXED;
    
    if (this.singleton != this.NOT_FIXED)
      return this.singleton;
    for (int v = 0; v < this.possibilities.length; v++)
      if (this.possibilities[v]) {
        count++;
        single = v;
      }
    if (count != 1)
      return this.NOT_FIXED;
    return single;
  }

  /**
   * Reset the cell.
   */
  public void reset() {
    int size = this.board.getSize();
    
    this.value = this.NOT_FIXED;
    this.singleton = this.NOT_FIXED;
    this.possibilities = new boolean[size];
    for (int i = 0; i < size; i++)
      this.possibilities[i] = true;
  }
  
  /**
   * Fix this cell.
   * 
   * @param value  The fixed value
   * 
   * @throws IllegalArgumentException if the value is not possible
   */
  public void fix(int value) throws IllegalStateException, IllegalArgumentException {
    assert !this.isFixed();
    if (value < 0 || value >= this.possibilities.length)
      throw new IllegalArgumentException(this.toString() + ": " + value + " is out of range");
    if (!this.possibilities[value])
      throw new IllegalArgumentException(this.toString() + ": " + value + " is not a possibility");
    this.getContext().trailFix(this);
    this.value = value;
    for (int i = 0; i < this.possibilities.length; i++) {
      if (i != value && this.possibilities[i]) {
        this.getContext().trailEliminate(this, i);
        this.possibilities[i] = false;
      }
    }
    for (Shape shape: this.shapes)
      shape.fixed(this);
    this.changed();
  }
  
  /**
   * Set this cell to be a singleton value.
   * 
   * @param value  The singleton value
   * 
   * @throws IllegalArgumentException if the value is not possible
   */
  public void singleton(int value) throws IllegalStateException, IllegalArgumentException {
    if (value < 0 || value >= this.possibilities.length)
      throw new IllegalArgumentException(this.toString() + ": " + value + " is out of range");
    if (!this.possibilities[value])
      throw new IllegalArgumentException(this.toString() + ": " + value + " is not a possibility");
    this.getContext().trailSingleton(this);
    this.singleton = value;
    this.changed();
 }
  
  /**
   * Eliminate a value from consideration.
  * 
   * @throws IllegalArgumentException if the value is not possible
   */
  public void eliminate(int value) {
    if (value < 0 || value >= this.possibilities.length)
      throw new IllegalArgumentException(this.toString() + ": " + value + " is out of range");
    if (this.isFixed())
      return;
    if (this.possibilities[value])
      this.getContext().trailEliminate(this, value);      
    this.possibilities[value] = false;
    for (Shape shape: this.shapes)
      shape.setSearch();
    this.changed();
  }
  
  /**
   * Rewind the fixing of this value.
   * 
   * @param trail The trail entry
   */
  public void rewind(TrailFix trail) {
    this.value = this.NOT_FIXED;
    this.changed();
  }
  
  /**
   * Rewind the singleton setting.
   * 
   * @param trail The trail entry
   */
  public void rewind(TrailSingleton trail) {
    this.singleton = this.NOT_FIXED;
    this.changed();
  }
  
  /**
   * Rewind the elimination of this value.
   * 
   * @param trail The trail entry
   */
  public void rewind(TrailEliminate trail) {
    this.possibilities[trail.getValue()] = true;
    this.changed();
  }
  
  /**
   * Trigger a search.
   * <p>
   * This is passed up to the parent board.
   */
  public void fireSearch() {
    this.board.search();
  }
}
